//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.4

import wesual.Controls 1.0

KeyEventSource {
    id : virtualKeyboard

    width  : content.implicitWidth
    height : content.implicitHeight

    property bool shift        : false
    property bool symbols      : false
    property int  symbolPage   : 0
    property bool enterEnabled : true

    Component {
        id : key

        Item {
            height : childrenRect.height
            width  : childrenRect.width
            Component.onCompleted :
                modelData.component.createObject(
                    this, modelData.data ? modelData.data : {})
        }
    }

    Component {
        id : letterKey

        VirtualKey {
            property var letters
            property int letterIndex : 0

            readonly property string text : {
                if (virtualKeyboard.shift) {
                    return letters[letterIndex].text.toUpperCase();
                }

                return letters[letterIndex].text;
            }

            Text {
                id : toggleText

                anchors {
                    centerIn : parent
                }
                font {
                    family : "Lato"
                    weight : Font.Light
                    pixelSize : 32
                }
                text  : parent.text
                color : highlight ? "white" : "#5F5F5F"

                Behavior on color {
                    ColorAnimation { duration : 70 }
                }

                Behavior on text {
                    SequentialAnimation {
                        NumberAnimation {
                            target   : toggleText
                            duration : 200
                            property : "opacity"
                            to        : 0
                        }
                        PropertyAction { property : "text" }
                        NumberAnimation {
                            target   : toggleText
                            duration : 200
                            property : "opacity"
                            to        : 1
                        }
                    }
                }
            }

            onClicked : virtualKeyboard.triggerKeyEvent(
                            letters[letterIndex].keyCode, text)
        }
    }

    Component {
        id : digitKey

        VirtualDigitKey {
            keyboard : virtualKeyboard
        }
    }

    Component {
        id : symbolKey

        VirtualSymbolKey {
            keyboard    : virtualKeyboard
            symbolIndex : virtualKeyboard.symbolPage
        }
    }

    Component {
        id : backspaceKey

        VirtualKey {
            height : 60
            width :  90

            onClicked : virtualKeyboard.triggerKeyEvent(Qt.Key_Backspace, "")

            ColorizedImage {
                anchors.centerIn : parent
                source           : "qrc:/controls/icons/key-backspace.png"
                color            : parent.highlight ? "white" : "#5F5F5F"

                Behavior on color {
                    ColorAnimation { duration : 70 }
                }
            }
        }
    }

    Component {
        id : shiftKey

        VirtualKey {
            height    : 60
            width     : 90
            pressed   : virtualKeyboard.shift
            onClicked : {
                virtualKeyboard.shift = !virtualKeyboard.shift;
            }

            ColorizedImage {
                anchors.centerIn : parent
                source           : "qrc:/controls/icons/key-shift.png"
                color            : parent.highlight ? "white" : "#5F5F5F"

                Behavior on color {
                    ColorAnimation { duration : 70 }
                }
            }
        }
    }

    Component {
        id : pageSwitchKey

        VirtualKey {
            height    : 60
            width     : 90

            ColorizedImage {
                id : pageSwitchIcon

                anchors.centerIn : parent
                source           : "qrc:/controls/icons/key-arrow.png"
                color            : parent.highlight ? "white" : "#5F5F5F"

                Behavior on color {
                    ColorAnimation { duration : 70 }
                }
            }

            states : [
                State {
                    name : "first"
                    when : virtualKeyboard.symbolPage !== 1

                    PropertyChanges {
                        target   : pageSwitchIcon
                        rotation : 0
                    }
                },
                State {
                    name : "second"
                    when : virtualKeyboard.symbolPage === 1

                    PropertyChanges {
                        target   : pageSwitchIcon
                        rotation : 180
                    }
                }
            ]

            transitions : [
                Transition {
                    to  : "second"

                    SequentialAnimation {
                        ParallelAnimation {
                            NumberAnimation {
                                target   : pageSwitchIcon
                                duration : 100
                                property : "opacity"
                                to        : 0
                            }
                            NumberAnimation {
                                target   : pageSwitchIcon
                                duration : 100
                                property : "anchors.horizontalCenterOffset"
                                to        : -30
                            }
                        }
                        PropertyAction {
                            target   : pageSwitchIcon
                            property : "rotation"
                        }
                        ParallelAnimation {
                            NumberAnimation {
                                target   : pageSwitchIcon
                                duration : 100
                                property : "opacity"
                                to        : 1
                            }
                            NumberAnimation {
                                target   : pageSwitchIcon
                                duration : 100
                                property : "anchors.horizontalCenterOffset"
                                from     : 30
                                to        : 0
                            }
                        }
                    }
                },
                Transition {
                    to  : "first"

                    SequentialAnimation {
                        ParallelAnimation {
                            NumberAnimation {
                                target   : pageSwitchIcon
                                duration : 100
                                property : "opacity"
                                to        : 0
                            }
                            NumberAnimation {
                                target   : pageSwitchIcon
                                duration : 100
                                property : "anchors.horizontalCenterOffset"
                                to        : 30
                            }
                        }
                        PropertyAction {
                            target   : pageSwitchIcon
                            property : "rotation"
                        }
                        ParallelAnimation {
                            NumberAnimation {
                                target   : pageSwitchIcon
                                duration : 100
                                property : "opacity"
                                to        : 1
                            }
                            NumberAnimation {
                                target   : pageSwitchIcon
                                duration : 100
                                property : "anchors.horizontalCenterOffset"
                                from     : -30
                                to        : 0
                            }
                        }
                    }
                }
            ]

            onClicked : virtualKeyboard.symbolPage =
                        virtualKeyboard.symbolPage === 1 ? 0 : 1
        }
    }

    readonly property var letterRows : [
        [
            // Q
            { component : letterKey, data : {
                    letters : [
                        { text : "q", keyCode : Qt.Key_Q }
                    ]
                }
            },
            // W
            { component : letterKey, data : {
                    letters : [
                        { text : "w", keyCode : Qt.Key_W }
                    ]
                }
            },
            // E
            { component : letterKey, data : {
                    letters : [
                        { text : "e", keyCode : Qt.Key_E }
                    ]
                }
            },
            // R
            { component : letterKey, data : {
                    letters : [
                        { text : "r", keyCode : Qt.Key_R }
                    ]
                }
            },
            // T
            { component : letterKey, data : {
                    letters : [
                        { text : "t", keyCode : Qt.Key_T }
                    ]
                }
            },
            // Z
            { component : letterKey, data : {
                    letters : [
                        { text : "z", keyCode : Qt.Key_Z }
                    ]
                }
            },
            // U
            { component : letterKey, data : {
                    letters : [
                        { text : "u", keyCode : Qt.Key_U }
                    ]
                }
            },
            // I
            { component : letterKey, data : {
                    letters : [
                        { text : "i", keyCode : Qt.Key_I }
                    ]
                }
            },
            // O
            { component : letterKey, data : {
                    letters : [
                        { text : "o", keyCode : Qt.Key_O }
                    ]
                }
            },
            // P
            { component : letterKey, data : {
                    letters : [
                        { text : "p", keyCode : Qt.Key_P }
                    ]
                }
            }
        ],
        [
            // A
            { component : letterKey, data : {
                    letters : [
                        { text : "a", keyCode : Qt.Key_A }
                    ]
                }
            },
            // P
            { component : letterKey, data : {
                    letters : [
                        { text : "s", keyCode : Qt.Key_S }
                    ]
                }
            },
            // D
            { component : letterKey, data : {
                    letters : [
                        { text : "d", keyCode : Qt.Key_D }
                    ]
                }
            },
            // F
            { component : letterKey, data : {
                    letters : [
                        { text : "f", keyCode : Qt.Key_F }
                    ]
                }
            },
            // G
            { component : letterKey, data : {
                    letters : [
                        { text : "g", keyCode : Qt.Key_G }
                    ]
                }
            },
            // H
            { component : letterKey, data : {
                    letters : [
                        { text : "h", keyCode : Qt.Key_H }
                    ]
                }
            },
            // J
            { component : letterKey, data : {
                    letters : [
                        { text : "j", keyCode : Qt.Key_J }
                    ]
                }
            },
            // K
            { component : letterKey, data : {
                    letters : [
                        { text : "k", keyCode : Qt.Key_K }
                    ]
                }
            },
            // L
            { component : letterKey, data : {
                    letters : [
                        { text : "l", keyCode : Qt.Key_L }
                    ]
                }
            }
        ],
        [
            // Shift
            { component : shiftKey },
            // Y
            { component : letterKey, data : {
                    letters : [
                        { text : "y", keyCode : Qt.Key_Y }
                    ]
                }
            },
            // X
            { component : letterKey, data : {
                    letters : [
                        { text : "x", keyCode : Qt.Key_X }
                    ]
                }
            },
            // C
            { component : letterKey, data : {
                    letters : [
                        { text : "c", keyCode : Qt.Key_C }
                    ]
                }
            },
            // V
            { component : letterKey, data : {
                    letters : [
                        { text : "v", keyCode : Qt.Key_V }
                    ]
                }
            },
            // B
            { component : letterKey, data : {
                    letters : [
                        { text : "b", keyCode : Qt.Key_B }
                    ]
                }
            },
            // N
            { component : letterKey, data : {
                    letters : [
                        { text : "n", keyCode : Qt.Key_N }
                    ]
                }
            },
            // M
            { component : letterKey, data : {
                    letters : [
                        { text : "m", keyCode : Qt.Key_M }
                    ]
                }
            },
            // Backspace
            { component : backspaceKey }
        ]
    ]

    readonly property var symbolRows : [
        [
            { component : digitKey, data : { digit : 1 } },
            { component : digitKey, data : { digit : 2 } },
            { component : digitKey, data : { digit : 3 } },
            { component : digitKey, data : { digit : 4 } },
            { component : digitKey, data : { digit : 5 } },
            { component : digitKey, data : { digit : 6 } },
            { component : digitKey, data : { digit : 7 } },
            { component : digitKey, data : { digit : 8 } },
            { component : digitKey, data : { digit : 9 } },
            { component : digitKey, data : { digit : 0 } }
        ],
        [
            // $^
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "$", keyCode : Qt.Key_Dollar },
                        { symbol : "^", keyCode : Qt.Key_acute  }
                    ]
                }
            },
            // %[
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "%", keyCode : Qt.Key_Percent },
                        { symbol : "[", keyCode : Qt.Key_BracketLeft }
                    ]
                }
            },
            // &]
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "&", keyCode : Qt.Key_Ampersand },
                        { symbol : "]", keyCode : Qt.Key_BracketRight }
                    ]
                }
            },
            // #{
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "#", keyCode : Qt.Key_brokenbar },
                        { symbol : "{", keyCode : Qt.Key_BraceLeft }
                    ]
                }
            },
            // *}
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "*", keyCode : Qt.Key_Asterisk },
                        { symbol : "}", keyCode : Qt.Key_BraceRight }
                    ]
                }
            },
            // (<
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "(",  keyCode : Qt.Key_ParenLeft },
                        { symbol : "<",  keyCode : Qt.Key_Less }
                    ]
                }
            },
            // )>
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : ")", keyCode : Qt.Key_ParentRight },
                        { symbol : ">", keyCode : Qt.Key_Greater     }

                    ]
                }
            },
            // -(
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "-", keyCode : Qt.Key_Minus },
                        { symbol : "§", keyCode : Qt.Key_paragraph }
                    ]
                }
            },
            // /)
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "/", keyCode : Qt.Key_Slash       },
                        { symbol : "€", keyCode : Qt.Key_currency }
                    ]
                }
            }
        ],
        [
            { component : pageSwitchKey },
            // !-
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "!", keyCode : Qt.Key_Exclam },
                        { symbol : "-", keyCode : Qt.Key_Minus }
                    ]
                }
            },
            // ;+
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : ";", keyCode : Qt.Key_Semicolon },
                        { symbol : "+", keyCode : Qt.Key_Plus }
                    ]
                }
            },
            // :=
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : ":", keyCode : Qt.Key_Colon },
                        { symbol : "=", keyCode : Qt.Key_Equal }
                    ]
                }
            },
            // '_
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "'",  keyCode : Qt.Key_Apostrophe },
                        { symbol : "_",  keyCode : Qt.Key_Underscore }
                    ]
                }
            },
            // "~
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "\"", keyCode : Qt.Key_QuoteDbl  },
                        { symbol : "~", keyCode : Qt.Key_AsciiTilde }

                    ]
                }
            },
            // ?|
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : "?", keyCode : Qt.Key_Question },
                        { symbol : "|", keyCode : Qt.Key_Bar }
                    ]
                }
            },
            // ,
            { component : symbolKey, data : {
                    symbols : [
                        { symbol : ",", keyCode : Qt.Key_Comma },
                        { symbol : ",", keyCode : Qt.Key_Comma }
                    ]
                }
            },
            { component : backspaceKey }
        ]
    ]

    Column {
        id : content

        spacing : 6

        states : [
            State {
                name  : "letters"
                when  : !virtualKeyboard.symbols

                PropertyChanges {
                    target  : letterKeys
                    enabled : true
                }
                PropertyChanges {
                    target  : symbolKeys
                    opacity : 0
                    enabled : false
                }
            },
            State {
                name : "symbols"
                when  : virtualKeyboard.symbols

                PropertyChanges {
                    target  : letterKeys
                    enabled : false
                }
                PropertyChanges {
                    target  : symbolKeys
                    opacity : 1
                    enabled : true
                }
            }
        ]

        transitions : Transition {
            NumberAnimation {
                property    : "opacity"
                duration    : 200
                easing.type : Easing.Linear
            }
        }

        Item {
            height : letterKeys.implicitHeight
            width  : letterKeys.implicitWidth

            // Letters
            Column {
                id : letterKeys

                spacing : 6

                Repeater {
                    model : letterRows

                    Row {
                        anchors.horizontalCenter : parent.horizontalCenter
                        spacing : 6

                        Repeater {
                            model    : modelData
                            delegate : key
                        }
                    }
                }
            }

            // Symbols
            Column {
                id : symbolKeys

                spacing : 6
                z       : 1

                Repeater {
                    model : symbolRows

                    Row {
                        anchors.horizontalCenter : parent.horizontalCenter
                        spacing : 6

                        Repeater {
                            model    : modelData
                            delegate : key
                        }
                    }
                }
            }
        }

        Row {
            spacing : 6
            anchors.horizontalCenter : parent.horizontalCenter

            VirtualKey {
                id : symbolToggleKey

                height    : 60
                width     : 90
                onClicked : {
                    virtualKeyboard.shift = false;
                    virtualKeyboard.symbolPage = 0;
                    virtualKeyboard.symbols = !virtualKeyboard.symbols
                }

                states : [
                    State {
                        name : "letters"
                        when : !virtualKeyboard.symbols

                        PropertyChanges {
                            target : toggleText
                            text   : "&123"
                        }
                    },
                    State {
                        name : "symbols"
                        when : virtualKeyboard.symbols

                        PropertyChanges {
                            target : toggleText
                            text   : "abc"
                        }
                    }
                ]

                transitions : [
                    Transition {
                        to  : "symbols"

                        SequentialAnimation {
                            ParallelAnimation {
                                NumberAnimation {
                                    target   : toggleText
                                    duration : 100
                                    property : "opacity"
                                    to        : 0
                                }
                                NumberAnimation {
                                    target   : toggleText
                                    duration : 100
                                    property : "anchors.horizontalCenterOffset"
                                    to        : -30
                                }
                            }
                            PropertyAction {
                                target   : toggleText
                                property : "text"
                            }
                            ParallelAnimation {
                                NumberAnimation {
                                    target   : toggleText
                                    duration : 100
                                    property : "opacity"
                                    to        : 1
                                }
                                NumberAnimation {
                                    target   : toggleText
                                    duration : 100
                                    property : "anchors.horizontalCenterOffset"
                                    from     : 30
                                    to        : 0
                                }
                            }
                        }
                    },
                    Transition {
                        to  : "letters"

                        SequentialAnimation {
                            ParallelAnimation {
                                NumberAnimation {
                                    target   : toggleText
                                    duration : 100
                                    property : "opacity"
                                    to        : 0
                                }
                                NumberAnimation {
                                    target   : toggleText
                                    duration : 100
                                    property : "anchors.horizontalCenterOffset"
                                    to        : 30
                                }
                            }
                            PropertyAction {
                                target   : toggleText
                                property : "text"
                            }
                            ParallelAnimation {
                                NumberAnimation {
                                    target   : toggleText
                                    duration : 100
                                    property : "opacity"
                                    to        : 1
                                }
                                NumberAnimation {
                                    target   : toggleText
                                    duration : 100
                                    property : "anchors.horizontalCenterOffset"
                                    from     : -30
                                    to        : 0
                                }
                            }
                        }
                    }
                ]

                Text {
                    id : toggleText

                    anchors {
                        centerIn : parent
                    }
                    font {
                        family : "Lato"
                        weight : Font.Light
                        pixelSize : 24
                    }
                    color : parent.highlight ? "white" : "#5F5F5F"

                    Behavior on color {
                        ColorAnimation {
                            duration : 70
                        }
                    }
                }
            }

            VirtualSymbolKey {
                symbols : [
                    { symbol : "@", keyCode : Qt.Key_At }
                ]
                keyboard : virtualKeyboard
            }

            VirtualKey {
                id : spaceKey

                height : 60
                width  : 324

                onClicked : virtualKeyboard.triggerKeyEvent(Qt.Key_Space, " ")
            }

            VirtualSymbolKey {
                symbols : [
                    { symbol : ".", keyCode : Qt.Key_Period }
                ]
                keyboard : virtualKeyboard
            }

            VirtualKey {
                id : enterKey

                height  : 60
                width   : 90
                enabled : virtualKeyboard.enterEnabled

                normalColor : "#90C695"

                onClicked : virtualKeyboard.triggerKeyEvent(Qt.Key_Enter, "")

                ColorizedImage {
                    anchors.centerIn : parent
                    source           : "qrc:/controls/icons/key-enter.png"
                    color            : "white"
                }
            }
        }
    }
}
